(function (window, undefined) {
    function TemplateXPathEngine() {
        const engine = this;
        if (!(engine instanceof TemplateXPathEngine)) {
            return new TemplateXPathEngine();
        }
    }

    TemplateXPathEngine.prototype.generateXpathByPatterns = function templateXPathEngine$generateXpathByPatterns(elementXPath, xpathDnd, hint, templates) {
        const CONSTANTS = {
            ERROR_MESSAGES: {
                ERROR_FETCHING_TARGET_ELEMENT_BY_XPATH: 'Target element could not be found with provided XPath $XPATH. Reason: $ERROR',
                ELEMENT_NOT_FOUND: 'Not found element with provided XPath',
                INVALID_OR_UNSUPPORTED_XPATH: 'Invalid or unsupported XPath: $XPATH',
                INVALID_XPATH: 'Invalid XPath',
                NO_MATCHING_TEMPLATE: 'No matching template found',
                MISSING_LABEL_TAG: '"$LABEL" used to define hints is missing from XPath template: $XPATH',
                MISSING_PATTERNS: 'Missing templates',
                MISSING_XPATH_EXPRESSION: 'Missing XPath expression',
                XPATH_RETURNS_SEVERAL_ELEMENTS: 'XPath returns several elements',
            },
            REPLACE_STRINGS: {
                AXE: '$AXE',
                ERROR: '$ERROR',
                LABEL: '$LABEL',
                VALUE: '$VALUE',
                XPATH: '$XPATH',
                WEIGHT: '$WEIGHT',
            },
            TAGS: {
                OPTION: 'option',
                SELECT: 'select',
            },
            XPATH_AXES: {
                ANCESTOR: 'ancestor',
                ANCESTOR_OR_SELF: 'ancestor-or-self',
                CHILD: 'child',
                DESCENDANT: 'descendant',
                DESCENDANT_OR_SELF: 'descendant-or-self',
                FOLLOWING: 'following',
                FOLLOWING_SIBLING: 'following-sibling',
                PARENT: 'parent',
                PRECEDING: 'preceding',
                PRECEDING_SIBLING: 'preceding-sibling',
            },
            XPATH_HELPERS: {
                AT: '@',
                AXES_NODE_DIVIDER: '::',
                AXE_PATTERN: '/$AXE::',
                BRACKETS_PATTERN: '[$VALUE]',
                CONTAINS: 'contains',
                DOT: '.',
                NODE_DIVIDER: '/',
                PARENTHESIS_PATTERN: '($VALUE)',
                TEXT_NODE_TAG: 'text()',
                START_XPATH: '//',
            },
            LOG_MESSAGES: {
                NO_REVERSE_XPATH: 'Failed to get reverse XPath',
                NO_LABELS: 'Not found labels with the given reverse XPath',
                NO_ELEMENTS_REVERSE_XPATH: 'No elements were found with the given reverse XPath',
                NO_ELEMENTS_XPATH: 'No elements with the generated XPath were found',
                INVALID_REVERSE_XPATH: 'Invalid reverse XPath',
                INVALID_XPATH: 'Invalid pattern XPath',
                NOT_MATCHED_LABELS: 'The Hint did not match any found labels',
                NOT_MATCHED_FOUNDED_ELEMENTS: 'None of the elements found from the generated XPath matched the target element',
                BIG_WEIGHT: 'Generated XPath weight is not optimal: $WEIGHT'
            },
            INDEX_WEIGHT: 2
        };
        const LOG_INFO = [];

        const getXpathWithHintByPatterns = (element, givenTemplates, hint) => {
            let result = [];
            let patternInfo = {};

            givenTemplates.forEach(template => {
                if (template.patterns && template.patterns.length > 0) {
                    template.patterns.forEach(pattern => {
                        patternInfo = {
                            id : pattern.id,
                            template: pattern.xpath,
                            weight: pattern.weight,
                            reverseXPath: null,
                            labels: [],
                            foundedElements: [],
                            xpath: null,
                            xpathWeight: null,
                            isMatched: false,
                            reason: null
                        };
                        let xPath;
                        let foundedElements;
                        let labels = findLabelsByPattern(pattern.xpath, element, patternInfo);
                        patternInfo.labels = labels ? labels : [];

                        if (labels) {
                            let isPatternForSelectOption = pattern.xpath.includes(CONSTANTS.TAGS.OPTION);
                            let isSelectElement = element.tagName.toLowerCase() === CONSTANTS.TAGS.SELECT && isPatternForSelectOption;
                            let isContains =
                                pattern.xpath.includes(CONSTANTS.XPATH_HELPERS.CONTAINS) ||
                                pattern.xpath.includes(CONSTANTS.XPATH_HELPERS.CONTAINS.toUpperCase());
                            let startIndex = pattern.xpath.indexOf(CONSTANTS.REPLACE_STRINGS.LABEL);
                            let notMatchedLabels = 0;
                            let notFoundedElementsByLabel = 0;
                            labels.forEach(label => {
                                let resultLabel = label;
                                let usedHint = hint;

                                if (usedHint !== null) {
                                    let sourceLabel = label;
                                    label = label.toLocaleLowerCase();

                                    let isWildCard = usedHint.startsWith('*');
                                    usedHint = isWildCard ?
                                        usedHint.slice(1).toLocaleLowerCase() :
                                        usedHint.toLocaleLowerCase();

                                    let indexOfHint = label.indexOf(usedHint);

                                    if (indexOfHint === -1) {
                                        notMatchedLabels++;
                                        return;
                                    }

                                    resultLabel = isWildCard ?
                                        sourceLabel :
                                        sourceLabel.substring(indexOfHint, indexOfHint + usedHint.length);
                                    if (isContains && isSelectElement) {
                                        resultLabel = resultLabel.toLocaleLowerCase();
                                    }
                                } else {
                                    if (isContains) {
                                        resultLabel = isSelectElement ?
                                            label.toLocaleLowerCase() :
                                            label.replace(new RegExp('\r?\n', 'gi'), '').trim();
                                    }
                                }

                                xPath = pattern.xpath.replace(
                                    CONSTANTS.REPLACE_STRINGS.LABEL,
                                    isContains && isSelectElement ? resultLabel.toLocaleLowerCase() : resultLabel
                                );

                                try {
                                    foundedElements = getElementsByXpath(xPath, targetDocument, null);
                                } catch (error) {
                                    notFoundedElementsByLabel++;
                                    return;
                                }
                                foundedElements.forEach((foundedElement, index) => {
                                    let xPathResult = {};
                                    if (foundedElement === element) {
                                        xPathResult = getXPathResult(xPath, resultLabel, result.length, index, pattern.id, pattern.weight);
                                        if (!isSelectElement) {
                                            result.push(xPathResult);
                                            patternInfo.xpath = xPathResult.xpath;
                                            patternInfo.xpathWeight = xPathResult.weight;
                                            patternInfo.isMatched = true;
                                        } else {
                                            let xPathLabel = xPath.substring(startIndex, startIndex + resultLabel.length);
                                            let optionText = element.selectedOptions[0].innerText;
                                            let condition = isContains ?
                                                xPathLabel.toLocaleLowerCase() === optionText.toLocaleLowerCase() :
                                                xPathLabel === optionText;
                                            if (condition) {
                                                result.push(xPathResult);
                                                patternInfo.xpath = xPathResult.xpath;
                                                patternInfo.xpathWeight = xPathResult.weight;
                                                patternInfo.isMatched = true;
                                            }
                                        }
                                    }

                                    patternInfo.foundedElements.push({
                                        element: foundedElement.outerHTML,
                                        xPath: xPathResult.xpath || xPath,
                                        xpathWeight: xPathResult.weight || pattern.weight
                                    });
                                });
                            });
                            if (labels.length === notMatchedLabels) {
                                patternInfo.reason = CONSTANTS.LOG_MESSAGES.NOT_MATCHED_LABELS;
                            } else if (labels.length === notFoundedElementsByLabel) {
                                patternInfo.reason = CONSTANTS.LOG_MESSAGES.NO_ELEMENTS_XPATH;
                            } else if (patternInfo.foundedElements.length !== 0 && !patternInfo.isMatched) {
                                patternInfo.reason = CONSTANTS.LOG_MESSAGES.NOT_MATCHED_FOUNDED_ELEMENTS;
                            }
                        } else {
                            if (!patternInfo.reason) {
                                patternInfo.reason = CONSTANTS.LOG_MESSAGES.NO_LABELS;
                            }
                        }

                        LOG_INFO.push({...patternInfo});
                        patternInfo = {};
                    });
                }
            });

            const sortedResult = result.sort((a, b) => a.weight - b.weight);

            LOG_INFO
                .filter(item => item.isMatched)
                .forEach((item, index) => {
                    if (item.id !== sortedResult[0].patternId) {
                        item.reason = CONSTANTS.LOG_MESSAGES.BIG_WEIGHT
                            .replace(CONSTANTS.REPLACE_STRINGS.WEIGHT, item.xpathWeight);
                    }
                });

            return sortedResult.length > 0 ? sortedResult[0] : null;
        };

        const findLabelsByPattern = (patternXPath, element, patternInfo) => {
            let labels;
            let labelElements;
            let reverseXPathObj;

            try {
                reverseXPathObj = buildReverseXPath(patternXPath, element);
                patternInfo.reverseXPath = reverseXPathObj.xPath;
                if (!reverseXPathObj.xPath) {
                    patternInfo.reason = CONSTANTS.LOG_MESSAGES.NO_REVERSE_XPATH;
                    return;
                }
                labelElements = getElementsByXpath(reverseXPathObj.xPath, targetDocument, element);
                labels = labelElements
                    .map(el => reverseXPathObj.attribute ?
                        el.attributes[reverseXPathObj.attribute] ?
                            el.attributes[reverseXPathObj.attribute].nodeValue : '' :
                        el.nodeType === Node.TEXT_NODE ?
                            el.data.replace(/\s/gi, ' ') : el.textContent || el.innerText.replace(/\s/gi, ' ')
                    )
                    .filter(label => label !== '')
                    .filter(label => label.trim().length);
            } catch (error) {
                switch (error.code) {
                    case 1:
                        patternInfo.reason = CONSTANTS.LOG_MESSAGES.NO_ELEMENTS_REVERSE_XPATH;
                        break;
                    case 2:
                        patternInfo.reason = CONSTANTS.LOG_MESSAGES.INVALID_REVERSE_XPATH;
                        break;
                    default:
                        patternInfo.reason = error.message;
                        break;
                }

                return;
            }

            return [...new Set(labels)];
        };

        const getXPathResult = (xPath, label, resultLength, index, patternId,patternWeight) => {
            let xpath = resultLength === 0 ?
                xPath :
                CONSTANTS.XPATH_HELPERS.PARENTHESIS_PATTERN.replace(CONSTANTS.REPLACE_STRINGS.VALUE, xPath) +
                CONSTANTS.XPATH_HELPERS.BRACKETS_PATTERN.replace(CONSTANTS.REPLACE_STRINGS.VALUE, (index + 1).toString());

            return {
                xpath: xpath,
                hint: label,
                patternId: patternId,
                weight: resultLength === 0 ? patternWeight : patternWeight + CONSTANTS.INDEX_WEIGHT
            };
        };

        const isValidXPath = expr => (
            typeof expr !== 'undefined' &&
            expr.replace(/[\s-_=]/g, '') !== ''
        );

        const getValidationRegex = () => {
            let regex =
                "(?P<node>" +
                "(" +
                "^id\\([\"\\']?(?P<idvalue>%(value)s)[\"\\']?\\)" +
                "|" +
                "(?P<axe>//?(?:%(axe)s)?)(?P<tag>%(tag)s)" +
                "(\\[(" +
                "(?P<matched>(?P<mattr>@?%(attribute)s=[\"\\'](?P<mvalue>%(value)s))[\"\\']" +
                "|" +
                "(?P<contained>contains\\((?P<cattr>@?%(attribute)s,\\s*[\"\\'](?P<cvalue>%(value)s)[\"\\']\\))" +
                ")\\])?" +
                "(\\[\\s*(?P<nth>\\d|last\\(\\s*\\))\\s*\\])?" +
                ")" +
                ")";

            const subRegexObj = {
                "axe": "(ancestor::)|(descendant::)|(following-sibling::)|(preceding-sibling::)|(child::)|(parent::)|(following::)|(preceding::)|(ancestor-or-self::)|(descendant-or-self::)",
                "tag": "([a-zA-Z][a-zA-Z0-9-][a-zA-Z-()]{0,35}|\\*)|([a-zA-Z][a-zA-Z0-9]{0,10}|\\*)",
                "attribute": "[.a-zA-Z_:][-\\w:.]*(\\(\\))?)",
                "value": "\\s*[\\w/:$][-/\\w\\s,:;.]*"
            };

            Object.keys(subRegexObj).forEach(key => {
                regex = regex.replace(new RegExp('%\\(' + key + '\\)s', 'gi'), subRegexObj[key]);
            });

            regex = regex.replace(/\?P<node>|\?P<idvalue>|\?P<axe>|\?P<tag>|\?P<matched>|\?P<mattr>|\?P<mvalue>|\?P<contained>|\?P<cattr>|\?P<cvalue>|\?P<nth>/gi, '');

            return new RegExp(regex, 'gi');
        };

        const preParseXpath = expr => (
            expr.replace(/contains\s*\(\s*concat\(["']\s+["']\s*,\s*@class\s*,\s*["']\s+["']\)\s*,\s*["']\s+([a-zA-Z0-9-_]+)\s+["']\)/gi, '@class="$1"')
        );

        const removeArraySigns = expr => {
            if (expr.startsWith('(')) {
                expr = expr
                    .substring(1)
                    .replace(/(\)\[(\d+)\])/gi, '');
            }
            return expr;
        };

        const changeDoubleSlashesToDescendantAxe = expr => (
            CONSTANTS.XPATH_HELPERS.START_XPATH
                .concat(
                    expr
                        .substring(2)
                        .replace(
                            /(\/\/)/gi,
                            CONSTANTS.XPATH_HELPERS.AXE_PATTERN
                                .replace(CONSTANTS.REPLACE_STRINGS.AXE, CONSTANTS.XPATH_AXES.DESCENDANT)
                        )
                )
        );

        const buildReverseXPath = (xPath, element) => {
            let reverseXpath = '';
            let errorMessage;

            if (!xPath.includes(CONSTANTS.REPLACE_STRINGS.LABEL)) {
                errorMessage = CONSTANTS.ERROR_MESSAGES.MISSING_LABEL_TAG
                    .replace(CONSTANTS.REPLACE_STRINGS.XPATH, xPath);
                console.log(`${new Date(Date.now()).toISOString()} -> Server Script Error: ${errorMessage}`);
                return {
                    xPath: null,
                    attribute: null
                };
            }

            if (!xPath) {
                errorMessage = CONSTANTS.ERROR_MESSAGES.MISSING_XPATH_EXPRESSION;
                console.log(`${new Date(Date.now()).toISOString()} -> Server Script Error: ${errorMessage}`);
                return {
                    xPath: null,
                    attribute: null
                };
            }

            xPath = preParseXpath(xPath);
            xPath = removeArraySigns(xPath);
            xPath = changeDoubleSlashesToDescendantAxe(xPath);

            if (!isValidXPath(xPath)) {
                errorMessage = CONSTANTS.ERROR_MESSAGES.INVALID_OR_UNSUPPORTED_XPATH
                    .replace(CONSTANTS.REPLACE_STRINGS.XPATH, xPath);
                console.log(`${new Date(Date.now()).toISOString()} -> Server Script Error: ${errorMessage}`);
                return {
                    xPath: null,
                    attribute: null
                };
            }

            const xPathArr = xPath.split('|');
            const validationRegex = getValidationRegex();
            let nodes = [];
            let xIndex = 0;

            while (xPathArr[xIndex]) {
                let execResults;

                while (execResults = validationRegex.exec(xPathArr[xIndex])) {
                    if (!execResults) {
                        errorMessage = CONSTANTS.ERROR_MESSAGES.INVALID_OR_UNSUPPORTED_XPATH
                            .replace(CONSTANTS.REPLACE_STRINGS.XPATH, xPath);
                        console.log(`${new Date(Date.now()).toISOString()} -> Server Script Error: ${errorMessage}`);
                        return {
                            xPath: null,
                            attribute: null
                        };
                    }

                    let node = {
                        idvalue: execResults[23] || execResults[3],
                        axe: execResults[4],
                        tag: execResults[15],
                        attrinfo: execResults[18],
                        matched: execResults[19],
                        mattr: execResults[21] || execResults[25],
                        mvalue: execResults[23] || execResults[27],
                        contained: execResults[24],
                        cattr: execResults[25],
                        cvalue: execResults[27],
                        nth: execResults[29]
                    };
                    node.xPath = node.axe.concat(node.tag, node.attrinfo ? node.attrinfo : '');
                    nodes.push(node);
                }
                xIndex++;
            }

            const nodeWithLabelIndex = nodes.findIndex(node => node.cvalue === CONSTANTS.REPLACE_STRINGS.LABEL || node.mvalue === CONSTANTS.REPLACE_STRINGS.LABEL);
            let attribute = null;
            let nodeWithLabel = null;
            if ((nodeWithLabelIndex !== 0 && nodeWithLabelIndex === nodes.length - 1) ||
                (nodeWithLabelIndex === 0 && nodes.length === 1)) {
                nodeWithLabel = nodes[nodeWithLabelIndex];
                nodes.forEach(node => {
                    reverseXpath += node.xPath;
                });

                if (nodeWithLabel.mattr[0] === CONSTANTS.XPATH_HELPERS.AT) {
                    attribute = nodeWithLabel.mattr.replace(CONSTANTS.XPATH_HELPERS.AT, '');
                    return {
                        xPath: reverseXpath.replace
                        (
                            nodeWithLabel.matched ? nodeWithLabel.matched : nodeWithLabel.contained,
                            nodeWithLabel.matched ? nodeWithLabel.mattr : nodeWithLabel.cattr
                        ),
                        attribute: attribute
                    };
                }
                if (nodeWithLabel.contained) {
                    return {
                        xPath: reverseXpath.replace(CONSTANTS.REPLACE_STRINGS.LABEL, ''),
                        attribute: attribute
                    };
                }
                if (nodeWithLabel.matched) {
                    return {
                        xPath: reverseXpath.replace(nodeWithLabel.attrinfo, ''),
                        attribute: attribute
                    };
                }
            } else {
                const reverseNodes = [...nodes].reverse();
                const elementName = reverseNodes[0].tag;
                if (elementName.toLowerCase() === element.tagName.toLowerCase()) {
                    reverseNodes[0].tag = CONSTANTS.XPATH_HELPERS.DOT;
                    let isChangeAxes = reverseNodes.some(node => node.axe.includes(CONSTANTS.XPATH_HELPERS.AXES_NODE_DIVIDER));
                    const axePatternAppliedToParent = CONSTANTS.XPATH_HELPERS.AXE_PATTERN
                        .replace(CONSTANTS.REPLACE_STRINGS.AXE, CONSTANTS.XPATH_AXES.PARENT);
                    nodeWithLabel = nodes[nodeWithLabelIndex];
                    if (nodeWithLabelIndex === 0 && nodeWithLabel.mattr[0] === CONSTANTS.XPATH_HELPERS.AT) {
                        attribute = nodeWithLabel.mattr.replace(CONSTANTS.XPATH_HELPERS.AT, '');
                    }
                    reverseNodes.forEach((node, index) => {
                        if (index !== reverseNodes.length - 1) {
                            if (index !== 0) {
                                if (!reverseNodes[index - 1].axe.includes(CONSTANTS.XPATH_HELPERS.AXES_NODE_DIVIDER)) {
                                    reverseXpath += axePatternAppliedToParent + node.tag + getReplacedAxe(node.axe);
                                } else {
                                    reverseXpath += node.tag + getReplacedAxe(node.axe);
                                }
                            } else {
                                reverseXpath += node.tag + getReplacedAxe(node.axe);
                            }
                        } else {
                            reverseXpath += isChangeAxes ?
                                node.tag === CONSTANTS.XPATH_HELPERS.TEXT_NODE_TAG ?
                                    node.tag + node.attrinfo.replace(CONSTANTS.REPLACE_STRINGS.LABEL, '') : node.tag :
                                axePatternAppliedToParent + node.tag;
                        }
                    });
                    return {
                        xPath: reverseXpath,
                        attribute: attribute
                    };
                }
            }

            return {
                xPath: null,
                attribute: null
            };
        };

        const getReplacedAxe = axe => {
            let result = CONSTANTS.XPATH_HELPERS.AXE_PATTERN;
            let replacedAxe;

            if (axe === CONSTANTS.XPATH_HELPERS.NODE_DIVIDER) {
                return axe;
            }

            const processedAxe = axe.match(/\/(.*?)::/)[1];

            switch (processedAxe) {
                case CONSTANTS.XPATH_AXES.ANCESTOR:
                    replacedAxe = CONSTANTS.XPATH_AXES.DESCENDANT;
                    break;
                case CONSTANTS.XPATH_AXES.DESCENDANT:
                    replacedAxe = CONSTANTS.XPATH_AXES.ANCESTOR;
                    break;
                case CONSTANTS.XPATH_AXES.FOLLOWING_SIBLING:
                    replacedAxe = CONSTANTS.XPATH_AXES.PRECEDING_SIBLING;
                    break;
                case CONSTANTS.XPATH_AXES.PRECEDING_SIBLING:
                    replacedAxe = CONSTANTS.XPATH_AXES.FOLLOWING_SIBLING;
                    break;
                case CONSTANTS.XPATH_AXES.CHILD:
                    replacedAxe = CONSTANTS.XPATH_AXES.PARENT;
                    break;
                case CONSTANTS.XPATH_AXES.PARENT:
                    replacedAxe = CONSTANTS.XPATH_AXES.CHILD;
                    break;
                case CONSTANTS.XPATH_AXES.FOLLOWING:
                    replacedAxe = CONSTANTS.XPATH_AXES.PRECEDING;
                    break;
                case CONSTANTS.XPATH_AXES.PRECEDING:
                    replacedAxe = CONSTANTS.XPATH_AXES.FOLLOWING;
                    break;
                case CONSTANTS.XPATH_AXES.ANCESTOR_OR_SELF:
                    replacedAxe = CONSTANTS.XPATH_AXES.DESCENDANT_OR_SELF;
                    break;
                case CONSTANTS.XPATH_AXES.DESCENDANT_OR_SELF:
                    replacedAxe = CONSTANTS.XPATH_AXES.ANCESTOR_OR_SELF;
                    break;
                default:
                    replacedAxe = axe;
                    break;
            }
            return result.replace(CONSTANTS.REPLACE_STRINGS.AXE, replacedAxe);
        };

        const getElementsByXpath = (xpath, targetDocument, contextNode) => {
            let result = targetDocument.evaluate(xpath, contextNode || targetDocument, null, XPathResult.ORDERED_NODE_ITERATOR_TYPE, null);
            if (result.invalidIteratorState) {
                throw {code: 2, message: CONSTANTS.ERROR_MESSAGES.INVALID_XPATH};
            }

            let elements = [];
            let element;
            while ((element = result.iterateNext()) !== null) {
                elements.push(element);
            }

            if (elements.length === 0) {
                throw {code: 1, message: CONSTANTS.ERROR_MESSAGES.ELEMENT_NOT_FOUND};
            } else {
                return elements;
            }
        };

        const getElementByXpath = (xpath, targetDocument) => {
            // PROOF-10955 -> Oracle JET FilePicker support
            if (xpath.includes('oj-file-picker')) {
                xpath = trimOracleJetXpath(xpath);
            }

            let result = targetDocument.evaluate(xpath, targetDocument, null, XPathResult.ORDERED_NODE_ITERATOR_TYPE, null);
            if (result.invalidIteratorState) {
                throw {code: 2, message: CONSTANTS.ERROR_MESSAGES.INVALID_XPATH};
            }

            let elements = [];
            let element;
            while ((element = result.iterateNext()) !== null) {
                elements.push(element);
            }

            if (elements.length === 0) {
                throw {code: 1, message: CONSTANTS.ERROR_MESSAGES.ELEMENT_NOT_FOUND};
            } else {
                if (elements.length === 1) {
                    return elements[0];
                } else {
                    throw {code: 0, message: CONSTANTS.ERROR_MESSAGES.XPATH_RETURNS_SEVERAL_ELEMENTS, elements: elements};
                }
            }
        };

        // PROOF-10955 -> Oracle JET FilePicker support
        const trimOracleJetXpath = (xpath) => {
            const match = /\/oj-file-picker(\[\d+\])?/.exec(xpath);

            if (match) {
                const index = match.index;
                const length = match[0].length;
                return xpath.substring(0, index + length);
            }

            return xpath;
        }

        let element;
        let targetDocument = document;
        let date = new Date();

        try {
            element = getElementByXpath(xpathDnd ? xpathDnd : elementXPath, targetDocument);
        } catch (error) {
            return {
                xpath: elementXPath,
                xpathDnd: xpathDnd,
                hint: hint,
                patternId: null,
                error: CONSTANTS.ERROR_MESSAGES.ERROR_FETCHING_TARGET_ELEMENT_BY_XPATH
                    .replace(CONSTANTS.REPLACE_STRINGS.XPATH, xpathDnd ? xpathDnd : elementXPath)
                    .replace(CONSTANTS.REPLACE_STRINGS.ERROR, error.message),
                logInfo: LOG_INFO,
                patternDate: date.toISOString()
            }
        }

        if (templates.length === 0 || templates.every(template => template.patterns.length === 0)) {
            return {
                xpath: elementXPath,
                xpathDnd: xpathDnd,
                hint: hint,
                patternId: null,
                error: CONSTANTS.ERROR_MESSAGES.MISSING_PATTERNS,
                logInfo: LOG_INFO,
                patternDate: date.toISOString()
            };
        }

        let xpathInfo = getXpathWithHintByPatterns(element, templates, hint);

        if (!xpathInfo) {
            return {
                xpath: elementXPath,
                xpathDnd: xpathDnd,
                hint: hint,
                patternId: null,
                error: CONSTANTS.ERROR_MESSAGES.NO_MATCHING_TEMPLATE,
                logInfo: LOG_INFO,
                patternDate: date.toISOString()
            };
        }

        return {
            xpath: xpathInfo.xpath === null ? elementXPath : xpathInfo.xpath,
            xpathDnd: xpathDnd,
            hint: xpathInfo.hint,
            patternId: xpathInfo.xpath === null ? null : xpathInfo.patternId,
            error: xpathInfo.xpath === null ? CONSTANTS.ERROR_MESSAGES.NO_MATCHING_TEMPLATE : null,
            logInfo: LOG_INFO,
            patternDate: date.toISOString()
        };
    }

    window.templateXPathEngine = new TemplateXPathEngine();
})(window);
